/****************************************************************************
 *   $Id:: pmu.c 9376 2012-04-19 22:59:58Z nxp41306                         $
 *   Project: NXP LPC11xxLV PMU example
 *
 *   Description:
 *     This file contains PMU code example which include PMU 
 *     initialization, PMU interrupt handler, and APIs for PMU
 *     reading.
 *
 ****************************************************************************
* Software that is described herein is for illustrative purposes only
* which provides customers with programming information regarding the
* products. This software is supplied "AS IS" without any warranties.
* NXP Semiconductors assumes no responsibility or liability for the
* use of the software, conveys no license or title under any patent,
* copyright, or mask work right to the product. NXP Semiconductors
* reserves the right to make changes in the software without
* notification. NXP Semiconductors also make no representation or
* warranty that such application will be suitable for the specified
* use without further testing or modification.

* Permission to use, copy, modify, and distribute this software and its 
* documentation is hereby granted, under NXP Semiconductors' 
* relevant copyright in the software, without fee, provided that it 
* is used in conjunction with NXP Semiconductors microcontrollers.  This 
* copyright, permission, and disclaimer notice must appear in all copies of 
* this code.
****************************************************************************/
#include "LPC11xxLV.h"			/* LPC11xxLV Peripheral Registers */
#include "type.h"
#include "gpio.h"
#include "pmu.h"

volatile uint32_t pmu_counter = 0;

/*****************************************************************************
** Function name:		WAKEUP_IRQHandler
**
** Descriptions:		WAKEUP Interrupt Handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void WAKEUP_IRQHandler(void)
{
  uint32_t regVal;

	LPC_GPIO[1]->MASKED_ACCESS[(1<<8)] = (1<<8);
	
  /* This handler takes care all the port pins if they
  are configured as wakeup source. */
  regVal = LPC_SYSCON->STARTSRP0;
  if ( regVal != 0 )
  {
	LPC_SYSCON->STARTRSRP0CLR = regVal;
  }
  
  //__NOP();
  return;
}

/*****************************************************************************
** Function name:		PMU_Init
**
** Descriptions:		Initialize PMU and setup wakeup source.
**						For Sleep and deepsleep, any of the I/O pins can be 
**						used as the wakeup source.
**						For Deep Powerdown, only pin P1.4 can be used as 
**						wakeup source from deep powerdown. 
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void PMU_Init( void )
{

   NVIC_EnableIRQ((IRQn_Type)(WAKEUP2_IRQn)); /* enable PIO0_2 as Interrupt source */
  
	/* use port0_2 as wakeup source, configure as input pin */
  GPIOSetDir( PORT0, 2, 0 );	/* Input P0.2 */
  LPC_IOCON->PIO0_2 &= ~0x07;	
  LPC_IOCON->PIO0_2 |= 0x20;	
   
  /* Only edge trigger. activation polarity on P1.0~11 and P1.0 is FALLING EDGE. */
  LPC_SYSCON->STARTAPRP0 = 0x00000000;
  /* Clear all wakeup source */ 
  LPC_SYSCON->STARTRSRP0CLR = 0xFFFFFFFF;
	
  /* Enable Port0.2 as wakeup source. */
  LPC_SYSCON->STARTERP0 = 0x4;
	
  return;
}

/*****************************************************************************
** Function name:		PMU_Sleep
**
** Descriptions:		Put some of the peripheral in sleep mode.
**
** parameters:			SleepMode: 1 is deep sleep, 0 is sleep, 
**						Sleep peripheral module(s)
** Returned value:		None
** 
*****************************************************************************/
void PMU_Sleep( uint32_t SleepMode, uint32_t SleepCtrl )
{
  LPC_SYSCON->PDAWAKECFG = LPC_SYSCON->PDRUNCFG;
  
  LPC_SYSCON->PDSLEEPCFG = SleepCtrl;
	
	/* If normal sleep, not deep sleep, don't do anything to SCR reg. */
  if ( SleepMode )
  {
	SCB->SCR |= NVIC_LP_SLEEPDEEP;
  }
  __WFI();
  return;
}

/******************************************************************************
**                            End Of File
******************************************************************************/
